//Class which adds a layer of abstraction and avoids having to worry about several issues:

// - "endian-ness":  When you create a BufferedFile for read or write, the first thing it does is read or
//    write a single byte flag to indicate the endianness of the file, and automatically converts to native
//    byte order if necessary when reading as long as you use the ReadNumber and WriteNumber functions.

// - size of numbers:  If I'm going down a structure with lots of members, writing each member, I find it
//   annoying to have to constantly check the type of each member, then call the appropriate sized byte
//   swap if necessary and write the appropriate size manually.  It is especially annoying when some members
//   are not base types (for example, time_t is not a base type - you have to know or look up to find out
//   that it is int64).  ReadNumber and WriteNumber accept any size or type of data (integers, floats, etc)
//   and treat it appropriately.  Thus, to write an array time_t TimeArray[5]
//      MyFile->WriteNumber(TimeArray,sizeof(time_t),5)

// - strings:  How do you write a string?  The length first, followed by the string, or omit the length
//   and search for the null terminator when you read?  BufferedFile provides ReadString() and WriteString()
//   to handle that for you.  No big deal but it simplifies things and keeps code more readable.

// - The last advantage is that it buffers the data internally during read and write, so that you can
//   conveniently read/write the data bits at a time (one member at a time, for example) without costly
//   repeated calls to the storage server.  Just using BFile, you should manually fill a write buffer for
//   the sake of efficiency and not call BFile::Read() or BFile::Write() too many times for small data
//   elements.  With buffering, it is just as efficient to write an int at a time.  This saves the hassle
//   by buffering output for you, allowing you to focus on the task at hand, namely writing your data one
//   piece at a time, and keeping your code simple and readable.  

//BufferedFile still provides all of the functionality of a BFile: It now provides Read/Write access, random
//access, etc.  All the regular BFile calls can be mixed with the BufferedFile calls, although the BFile calls
//won't benefit from the buffering (they force the buffer to flush, and actually introduce inefficiency - use
//them carefully; in general try to stick to the BufferedFile-specific calls)


//******************************************************************************************************
//**** SYSTEM HEADER FILES
//******************************************************************************************************
#include <stdio.h>
#include <string.h>


//******************************************************************************************************
//**** PROJECT HEADER FILES
//******************************************************************************************************
#include "BufferedFile.h"


//******************************************************************************************************
//**** BufferedFile CLASS DEFINITION
//******************************************************************************************************
BufferedFile::BufferedFile(const entry_ref* ref, uint32 open_Mode, int32 Buffer_Size)
: BFile(ref,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(Buffer_Size);
}


BufferedFile::BufferedFile(const BEntry* entry, uint32 open_Mode, int32 Buffer_Size)
: BFile(entry,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(Buffer_Size);
}


BufferedFile::BufferedFile(const char* path, uint32 open_Mode, int32 Buffer_Size)
: BFile(path,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(Buffer_Size);
}


BufferedFile::BufferedFile(BDirectory* dir, const char* path, uint32 open_Mode, int32 Buffer_Size)
: BFile(dir,path,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(Buffer_Size);
}


void BufferedFile::CheckNewFileStatus()
{
	//Check the status of the file
	Valid = BFile::InitCheck();
}


void BufferedFile::AllocAndInitBufferAndEndian(int32 Buffer_Size)
{
	Buffer = NULL;
	if(Valid == B_NO_ERROR)
	{
		//Allocate the buffer
		BufferSize = Buffer_Size;
		Buffer = new char[BufferSize];
		BytesInBuffer = 0;
		BufferPos = 0;
		WriteBuffered = false;

		//Set the native endianness for easy comparison later
		#if defined(__POWERPC__)
			//PPC: Big endian
			HostBigEndian = true;
		#else
		#if defined(__INTEL__)
			//INTEL: Little endian
			HostBigEndian = false;
		#else
			//NOT PPC OR INTEL:  - unknown future processor  (Alpha!  Woohoo!)
			***Generate a compile time error***;
		#endif
		#endif

		//Set (write) or check (read) the endianness recorded in the file
		status_t Result = B_NO_ERROR;
		bool BigEndianSet = false;
		if(openMode & B_OPEN_AT_END)
		{
			Result = BFile::Seek(0,SEEK_SET);
			if(Result != B_NO_ERROR)
			{
				Valid = Result;
				return;
			}
		}

		if(  (openMode&O_RWMASK)==B_READ_ONLY ||
			((openMode&O_RWMASK)==B_READ_WRITE && !(openMode & B_ERASE_FILE)) ||
			((openMode&O_RWMASK)==B_WRITE_ONLY && (openMode&B_OPEN_AT_END) && !(openMode&B_ERASE_FILE)))
		{
			if(BFile::Read(&BigEndian,sizeof(BigEndian)) == sizeof(BigEndian))
				BigEndianSet = true;
		}
		if(!BigEndianSet && ((openMode&O_RWMASK)==B_WRITE_ONLY || (openMode&O_RWMASK)==B_READ_WRITE))
		{
			BigEndian = HostBigEndian;
			if(BFile::Write(&BigEndian,sizeof(BigEndian)) == sizeof(BigEndian))
				BigEndianSet = true;
		}
		if(!BigEndianSet)
		{
			Valid = B_FILE_ERROR;
			return;
		}
		if(openMode & B_OPEN_AT_END)
		{
			Result = BFile::Seek(0,SEEK_END);
			if(Result != B_NO_ERROR)
			{
				Valid = Result;
				return;
			}
		}
	}
}


BufferedFile::~BufferedFile()
{
	if(Buffer)
		delete[] Buffer;
}


status_t BufferedFile::FlushBuffer()
//Flushes the read buffer and moves the file pointer to the position that was being read from.  It is
//automatically called when necessary when working within the file, but if you have buffered write data, call
//it yourself before deleting the BufferedFile to make sure the data is written successfully (the destructor
//can't return an error code).
{
	if(WriteBuffered)
	{
		if(BFile::Write(Buffer,BufferPos) != BufferPos)
			return B_FILE_ERROR;
		BufferPos = 0;
		WriteBuffered = false;
	}
	else if(BufferPos != 0)
	{
		off_t Result = BFile::Seek(off_t(BufferPos)-off_t(BytesInBuffer),SEEK_CUR);
		if(Result == B_ERROR || Result == B_FILE_ERROR)
			return B_FILE_ERROR;
		BufferPos = 0;
	}
	return B_NO_ERROR;
}


status_t BufferedFile::InitCheck()
//Returns B_NO_ERROR if the BufferedFile was created successfully.  Returns B_NO_INIT if initialization of
//the underlying BFile failed.  Returns B_ERROR if reading or writing the endianness flag to or from the
//first byte of the file failed.
{
	return Valid;
}


int32 BufferedFile::WriteRaw(const void* WriteBuffer, size_t BytesToWrite)
//Writes the raw data contained in WriteBuffer.  Returns the number of bytes actually written.
{
	if(!IsWritable())
		return 0;
	size_t Counter = 0;
	if(!WriteBuffered && BufferPos != 0)
		if(FlushBuffer() != B_NO_ERROR)
			return 0;
	while(Counter < BytesToWrite)
	{
		Buffer[BufferPos++] = ((const char*)WriteBuffer)[Counter++];
		if(BufferPos == BufferSize)
		{
			//Buffer is full, need to write and reset
			int32 Result = BFile::Write(Buffer,BufferSize);
			if(Result != BufferSize)
			{
				WriteBuffered = false;
				return Counter-(BufferSize-Result);
			}
			BufferPos = 0;
		}
	}
	if(BufferPos > 0)
		WriteBuffered = true;
	return Counter;
}


int32 BufferedFile::ReadRaw(void* ReadBuffer, size_t BytesToRead)
//Reads raw data into ReadBuffer.  Returns the number of bytes actually read.
{
	if(!IsReadable())
		return 0;
	if(WriteBuffered)
		if(FlushBuffer() != B_NO_ERROR)
			return 0;
	size_t Counter = 0;
	while(Counter < BytesToRead)
	{
		if(BufferPos == 0)
		{
			ssize_t Result = BFile::Read(Buffer,BufferSize);
			BytesInBuffer = Result;
			if(Result < ssize_t(BufferSize))
			{
				if(Result <= 0)
					return Counter;
			}
		}
		((char*)ReadBuffer)[Counter++] = ((const char*)Buffer)[BufferPos++];
		if(BufferPos == BytesInBuffer)
		{
			//Reached end of buffer, need to reset for another read
			if(BytesInBuffer != BufferSize)
			{
				//Already read to the end of the file
				BufferPos = 0;
				BytesInBuffer = 0;
				return Counter;
			}
			BufferPos = 0;
			BytesInBuffer = 0;
		}
	}
	return Counter;
}


status_t BufferedFile::ReadString(char* StringBuffer,int32 BufferSize)
//Reads a string from the specified file.  Returns B_NO_ERROR if successful.  The string will be null-
//terminated.  Failure could be due to reaching the end of the file before finishing the read (returns
//B_FILE_ERROR), or due to running out of buffer space (returns B_ERROR).
{
	int32 Counter = 0;
	do
	{
		if(ReadRaw(&StringBuffer[Counter],1) != 1)
		{
			StringBuffer[Counter] = 0;
			return B_FILE_ERROR;
		}
		Counter++;
	}while(StringBuffer[Counter-1] != 0 && Counter < BufferSize);
	if(Counter == BufferSize && StringBuffer[BufferSize-1] != 0)
	{
		StringBuffer[BufferSize-1] = 0;
		return B_ERROR;
	}
	return B_NO_ERROR;
}


status_t BufferedFile::WriteString(const char* StringBuffer)
//Writes a string.  The string must be null-terminated.  Returns B_NO_ERROR if successful, B_FILE_ERROR if
//it fails.  If passed NULL, it writes a zero byte (empty, NULL-terminated string) to the file.
{
	if(StringBuffer == NULL)
		StringBuffer = "";
	int32 StringLength = strlen(StringBuffer)+1;			//Includes null terminator
	if(WriteRaw(StringBuffer,StringLength) != StringLength)
		return B_FILE_ERROR;
	else
		return B_NO_ERROR;
}


int32 BufferedFile::ReadNumber(void* ReadBuffer, uint8 SizeOfData, int32 NumbersToRead)
//Read a number without regard to size, in native-endian format.  SizeOfData is the size of EACH data
//element in bytes, not the whole buffer.  NumbersToRead is the number of data elements to read, NOT the
//size of the whole buffer in bytes.  Example usage:
//   time_t TimeArray[5];
//   MyFile->ReadNumber(TimeArray,sizeof(time_t),5)
{
	if(SizeOfData != 1 && SizeOfData != 2 && SizeOfData != 4 && SizeOfData != 8)
		return 0;
	int32 Result = ReadRaw(ReadBuffer,SizeOfData*NumbersToRead) / SizeOfData;
	//Swap byte order if necessary
	if(BigEndian != HostBigEndian)
	{
		for(int32 ReadIndex = 0; ReadIndex < Result; ReadIndex++)
		{
			for(uint8 Counter = 0; Counter < SizeOfData/2; Counter++)
			{
				char Temp = ((char*)ReadBuffer)[ReadIndex*SizeOfData + Counter];
				((char*)ReadBuffer)[ReadIndex*SizeOfData + Counter] =
					((char*)ReadBuffer)[ReadIndex*SizeOfData + SizeOfData-Counter-1];
				((char*)ReadBuffer)[ReadIndex*SizeOfData + SizeOfData-Counter-1] = Temp;
			}
		}
	}
	return Result;
}


int32 BufferedFile::WriteNumber(void* WriteBuffer, uint8 SizeOfData, int32 NumbersToWrite)
//Write a number without regard to size, in native-endian format.  SizeOfData is the size of EACH data
//element in bytes, not the whole buffer.  NumbersToWrite is the number of data elements to read, NOT the
//size of the whole buffer in bytes.  Example usage:
//   time_t TimeArray[5] = {0LL,0LL,0LL,0LL,0LL};
//   MyFile->WriteNumber(TimeArray,sizeof(time_t),5)
{
	if(SizeOfData != 1 && SizeOfData != 2 && SizeOfData != 4 && SizeOfData != 8)
		return 0;
	//Swap byte order if necessary
	if(BigEndian != HostBigEndian)
	{
		for(int32 WriteIndex = 0; WriteIndex < NumbersToWrite; WriteIndex++)
		{
			for(uint8 Counter = 0; Counter < SizeOfData/2; Counter++)
			{
				char Temp = ((char*)WriteBuffer)[WriteIndex*SizeOfData + Counter];
				((char*)WriteBuffer)[WriteIndex*SizeOfData + Counter] =
					((char*)WriteBuffer)[WriteIndex*SizeOfData + SizeOfData-Counter-1];
				((char*)WriteBuffer)[WriteIndex*SizeOfData + SizeOfData-Counter-1] = Temp;
			}
		}
	}
	int32 Result = WriteRaw(WriteBuffer,SizeOfData*NumbersToWrite)/SizeOfData;
	//Swap byte order back if necessary
	if(BigEndian != HostBigEndian)
	{
		for(int32 WriteIndex = 0; WriteIndex < NumbersToWrite; WriteIndex++)
		{
			for(uint8 Counter = 0; Counter < SizeOfData/2; Counter++)
			{
				char Temp = ((char*)WriteBuffer)[WriteIndex*SizeOfData + Counter];
				((char*)WriteBuffer)[WriteIndex*SizeOfData + Counter] =
					((char*)WriteBuffer)[WriteIndex*SizeOfData + SizeOfData-Counter-1];
				((char*)WriteBuffer)[WriteIndex*SizeOfData + SizeOfData-Counter-1] = Temp;
			}
		}
	}
	return Result;
}


bool BufferedFile::IsBigEndian()
//Tells whether the file being read is BigEndian or not
{
	return BigEndian;
}


bool BufferedFile::IsNativeEndian()
//Tells whether the file being read is already in native endian format
{
	return HostBigEndian == BigEndian;
}


ssize_t BufferedFile::Read(void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::Read(buffer,size);
}


ssize_t BufferedFile::ReadAt(off_t pos, void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::ReadAt(pos,buffer,size);
}


ssize_t BufferedFile::Write(const void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::Write(buffer,size);
}


ssize_t BufferedFile::WriteAt(off_t pos, const void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::WriteAt(pos,buffer,size);
}


off_t BufferedFile::Seek(off_t position, uint32 seek_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::Seek(position,seek_mode);
}


status_t BufferedFile::SetSize(off_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	return BFile::SetSize(size);
}


status_t BufferedFile::SetTo(const entry_ref *ref, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	if(Buffer)
	{
		delete[] Buffer;
		Buffer = NULL;
	}
	Valid = BFile::SetTo(ref,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(BufferSize);
	return Valid;
}


status_t BufferedFile::SetTo(const BEntry *entry, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	if(Buffer)
	{
		delete[] Buffer;
		Buffer = NULL;
	}
	Valid = BFile::SetTo(entry,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(BufferSize);
	return Valid;
}


status_t BufferedFile::SetTo(const char *path, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	if(Buffer)
	{
		delete[] Buffer;
		Buffer = NULL;
	}
	Valid = BFile::SetTo(path,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(BufferSize);
	return Valid;
}


status_t BufferedFile::SetTo(const BDirectory *dir, const char *path, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	if(Buffer)
	{
		delete[] Buffer;
		Buffer = NULL;
	}
	Valid = BFile::SetTo(dir,path,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	AllocAndInitBufferAndEndian(BufferSize);
	return Valid;
}
